<?php
/*
Plugin Name: LCK cloud Connector
Plugin URI: https://www.lck-cloud.jp/membership-site-creation-guide.html
Description: Official connector plugin to securely link WordPress and LCK cloud using SaaS Bridge Architecture. Isolates member management and payments to the cloud for enhanced security.
Version: 1.0.4
Author: LCK cloud (Telecommunications Business Operator E-02-04640)
Author URI: https://www.lck-cloud.jp/about.html
License: GPL2
Text Domain: lck-cloud-connector
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Lck_Cloud_Connector {

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_plugin_page' ) );
		add_action( 'admin_init', array( $this, 'page_init' ) );
		add_action( 'template_redirect', array( $this, 'execute_lck_cloud' ) );
	}

	public function add_plugin_page() {
		add_options_page(
			'LCK cloud 設定',
			'LCK cloud',
			'manage_options',
			'lck-cloud-setting',
			array( $this, 'create_admin_page' )
		);
	}

	public function create_admin_page() {
		?>
		<div class="wrap">
			<h1>LCK cloud Connector 設定</h1>
			<form method="post" action="options.php">
				<?php
				settings_fields( 'lck_cloud_option_group' );
				do_settings_sections( 'lck-cloud-setting' );
				submit_button();
				?>
			</form>
			<hr>
			<p>※設定方法の詳細は <a href="https://www.lck-cloud.jp/membership-site-creation-guide.html" target="_blank">こちら</a> を参照してください。</p>
		</div>
		<?php
	}

	public function page_init() {
		register_setting(
			'lck_cloud_option_group',
			'lck_cloud_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
			)
		);

		add_settings_section( 'setting_section_id', '基本設定', null, 'lck-cloud-setting' );
		add_settings_field( 'lck_cloud_uno', '登録番号 (uno)', array( $this, 'uno_callback' ), 'lck-cloud-setting', 'setting_section_id' );
		add_settings_field( 'lck_cloud_domain', 'LCKドメイン', array( $this, 'domain_callback' ), 'lck-cloud-setting', 'setting_section_id' );
		add_settings_field( 'lck_cloud_error', 'エラー遷移先URL', array( $this, 'error_callback' ), 'lck-cloud-setting', 'setting_section_id' );
		add_settings_field( 'lck_set', 'アクセス制限ルール (1行ずつ入力)', array( $this, 'set_callback' ), 'lck-cloud-setting', 'setting_section_id' );
	}

	public function sanitize_settings( $input ) {
		$new_input = array();
		if ( isset( $input['lck_cloud_uno'] ) ) {
			$new_input['lck_cloud_uno'] = sanitize_text_field( $input['lck_cloud_uno'] );
		}
		if ( isset( $input['lck_cloud_domain'] ) ) {
			$new_input['lck_cloud_domain'] = sanitize_text_field( $input['lck_cloud_domain'] );
		}
		if ( isset( $input['lck_cloud_error'] ) ) {
			$new_input['lck_cloud_error'] = esc_url_raw( $input['lck_cloud_error'] );
		}
		if ( isset( $input['lck_set'] ) ) {
			$new_input['lck_set'] = sanitize_textarea_field( $input['lck_set'] );
		}
		return $new_input;
	}

	public function uno_callback() {
		$options = get_option( 'lck_cloud_settings' );
		printf( '<input type="text" name="lck_cloud_settings[lck_cloud_uno]" value="%s" class="regular-text" />', isset( $options['lck_cloud_uno'] ) ? esc_attr( $options['lck_cloud_uno'] ) : '' );
	}

	public function domain_callback() {
		$options = get_option( 'lck_cloud_settings' );
		printf( '<input type="text" name="lck_cloud_settings[lck_cloud_domain]" value="%s" class="regular-text" />', isset( $options['lck_cloud_domain'] ) ? esc_attr( $options['lck_cloud_domain'] ) : 'fs.lck-cloud.com' );
	}

	public function error_callback() {
		$options = get_option( 'lck_cloud_settings' );
		printf( '<input type="text" name="lck_cloud_settings[lck_cloud_error]" value="%s" class="regular-text" placeholder="https://example.com/error/" />', isset( $options['lck_cloud_error'] ) ? esc_attr( $options['lck_cloud_error'] ) : '' );
		echo '<p class="description">ワードプレス側のURL（自ドメイン内）を記入してください。</p>';
	}

	public function set_callback() {
		$options = get_option( 'lck_cloud_settings' );
		$val     = isset( $options['lck_set'] ) ? $options['lck_set'] : '';
		// Escaping needs to happen at the point of output
		echo '<textarea name="lck_cloud_settings[lck_set]" rows="5" cols="50" class="regular-text" placeholder="/member/,group_id">' . esc_textarea( $val ) . '</textarea>';
	}

	public function execute_lck_cloud() {
		if ( is_admin() ) {
			return;
		}
		$options = get_option( 'lck_cloud_settings' );
		if ( empty( $options['lck_cloud_uno'] ) ) {
			return;
		}

		$lck_cloud_uno    = $options['lck_cloud_uno'];
		$lck_cloud_domain = $options['lck_cloud_domain'];
		$lck_cloud_error  = $options['lck_cloud_error'];
		$lck_cloud_path   = '/';
		$lck_set_raw      = isset( $options['lck_set'] ) ? explode( "\n", str_replace( "\r", '', $options['lck_set'] ) ) : array();

		$f           = 0;
		$grp         = '';
		$current_uri = '';

		// Verify and Sanitize SERVER variable
		if ( isset( $_SERVER['REQUEST_URI'] ) ) {
			$current_uri = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) );
		}

		foreach ( $lck_set_raw as $line ) {
			$line = trim( $line );
			if ( empty( $line ) ) {
				continue;
			}
			$v2          = explode( ',', $line, 2 );
			$target_path = trim( $v2[0] );
			$target_grp  = ( isset( $v2[1] ) ) ? trim( $v2[1] ) : '';
			$pattern     = preg_quote( $target_path, '/' );
			if ( preg_match( '/' . $pattern . '/', $current_uri ) ) {
				$f   = 1;
				$grp = $target_grp;
				break;
			}
		}

		if ( ! $f ) {
			return;
		}

		$http_host = '';
		if ( isset( $_SERVER['HTTP_HOST'] ) ) {
			$http_host = sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) );
		}

		// Generate a return URL
		$base_url = ( is_ssl() ? 'https' : 'http' ) . '://' . $http_host . $current_uri;
		// Adding WordPress Nonce to URL
		$r = add_query_arg( '_lck_nonce', wp_create_nonce( 'lck_cloud_action' ), $base_url );
		$cin = 0;
		$cid = '';

		// Safe retrieval of Cookie
		if ( isset( $_COOKIE['lck_cloud_cid'] ) ) {
			$cid = sanitize_text_field( wp_unslash( $_COOKIE['lck_cloud_cid'] ) );
		}

		// Verify the nonce returned from the external service.
		if ( isset( $_GET['lck_cloud_in'] ) ) {
			$nonce = isset( $_GET['_lck_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_lck_nonce'] ) ) : '';
			// Verify that the nonce "lck_cloud_action" is valid.
			if ( ! wp_verify_nonce( $nonce, 'lck_cloud_action' ) ) {
				// Stop execution if security check fails.
				wp_die( 'Security check failed. Please try again.' );
			}
			$cin = 1;
			$cid = sanitize_text_field( wp_unslash( $_GET['lck_cloud_in'] ) );
			setcookie( 'lck_cloud_cid', $cid, 0, $lck_cloud_path );
		}

		$api_url = 'https://' . $lck_cloud_domain . '/lck.html?cin=' . $cin . '&cid=' . $cid . '&grp=' . $grp . '&r=' . urlencode( $r ) . '&uno=' . $lck_cloud_uno;

		// Enable SSL verification (use the default WordPress settings)
		$response = wp_remote_get( $api_url );
		if ( is_wp_error( $response ) ) {
			wp_die( 'LCK cloud Access Error' );
		}

		$fg = wp_remote_retrieve_body( $response );

		if ( substr( $fg, 0, 4 ) == 'http' ) {
			// Validate the URL before redirecting
			$redirect_url = wp_http_validate_url( $fg );
			if ( $redirect_url ) {
				// We are redirecting to an external API-provided URL, so safe_redirect might block it.
				// However, we have validated it is a URL.
				// phpcs:ignore WordPress.Security.SafeRedirect.wp_redirect_wp_redirect
				wp_redirect( $redirect_url );
				exit;
			}
		}

		if ( $fg != 1 ) {
			wp_safe_redirect( $lck_cloud_error );
			exit;
		}
	}
}
new Lck_Cloud_Connector();
